// Content script for Clip2Post extension
// Captures text selection and sends to background worker

(function () {
    'use strict';

    // Debounce function to avoid excessive messaging
    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    // Capture selection and send to background
    const captureSelection = debounce(() => {
        const selection = window.getSelection();
        const text = selection?.toString().trim();

        if (text && text.length > 0) {
            chrome.runtime.sendMessage(
                { type: 'SELECTION_CAPTURED', text },
                (response) => {
                    if (chrome.runtime.lastError) {
                        // Extension context invalidated, ignore
                        return;
                    }
                }
            );
        }
    }, 300);

    // Listen for mouseup events to capture selection
    document.addEventListener('mouseup', (e) => {
        // Small delay to ensure selection is complete
        setTimeout(captureSelection, 10);
    });

    // Listen for keyboard selection (Shift + Arrow keys)
    document.addEventListener('keyup', (e) => {
        if (e.shiftKey) {
            captureSelection();
        }
    });
})();
