// API client for Clip2Post extension
// Token-based authentication for MV3 compatibility

const API_BASE = 'https://click2post.io'; // Production URL

class Clip2PostAPI {
    constructor() {
        this.isAuthenticated = false;
        this.token = null;
    }

    // Get token from chrome.storage
    async getToken() {
        return new Promise((resolve) => {
            chrome.storage.local.get(['authToken'], (result) => {
                console.log('[API] Retrieved token from storage:', result.authToken ? 'exists' : 'null');
                resolve(result.authToken || null);
            });
        });
    }

    // Save token to chrome.storage
    async saveToken(token) {
        return new Promise((resolve) => {
            chrome.storage.local.set({ authToken: token }, () => {
                console.log('[API] Token saved to storage');
                this.token = token;
                resolve();
            });
        });
    }

    // Remove token from chrome.storage
    async removeToken() {
        return new Promise((resolve) => {
            chrome.storage.local.remove(['authToken'], () => {
                console.log('[API] Token removed from storage');
                this.token = null;
                resolve();
            });
        });
    }

    async checkAuth() {
        try {
            const token = await this.getToken();
            if (!token) {
                console.log('[API] No token found, user not authenticated');
                this.isAuthenticated = false;
                return null;
            }

            console.log('[API] Checking auth with token...');
            const response = await fetch(`${API_BASE}/api/me`, {
                headers: {
                    'Authorization': `Bearer ${token}`,
                },
            });

            if (response.ok) {
                const data = await response.json();
                console.log('[API] Auth check successful:', data.email);
                this.isAuthenticated = true;
                this.token = token;
                return data;
            } else {
                console.log('[API] Auth check failed, status:', response.status);
                this.isAuthenticated = false;
                await this.removeToken();
                return null;
            }
        } catch (error) {
            console.error('[API] Auth check error:', error);
            this.isAuthenticated = false;
            return null;
        }
    }

    async request(endpoint, options = {}) {
        const token = await this.getToken();

        const headers = {
            'Content-Type': 'application/json',
            ...options.headers,
        };

        if (token) {
            headers['Authorization'] = `Bearer ${token}`;
        }

        console.log(`[API] Request: ${options.method || 'GET'} ${endpoint}`);

        const response = await fetch(`${API_BASE}${endpoint}`, {
            ...options,
            headers,
        });

        if (response.status === 401) {
            console.log('[API] 401 Unauthorized, clearing token');
            this.isAuthenticated = false;
            await this.removeToken();
            throw new Error('Session expired. Please sign in again.');
        }

        const data = await response.json();

        if (!response.ok) {
            console.error('[API] Request failed:', data.error);
            throw new Error(data.error || 'Request failed');
        }

        console.log('[API] Request successful');
        return data;
    }

    async getMe() {
        return this.request('/api/me');
    }

    async generate(platform, inputText) {
        return this.request('/api/generate', {
            method: 'POST',
            body: JSON.stringify({ platform, inputText }),
        });
    }

    async signIn(email, password) {
        console.log('[API] Signing in...', email);
        const data = await this.request('/api/auth', {
            method: 'POST',
            body: JSON.stringify({ email, password, action: 'signin' }),
        });

        // Save the token from the response
        if (data.token) {
            console.log('[API] Sign in successful, saving token');
            await this.saveToken(data.token);
            this.isAuthenticated = true;
        }

        return data;
    }

    async signUp(email, password) {
        console.log('[API] Signing up...', email);
        const data = await this.request('/api/auth', {
            method: 'POST',
            body: JSON.stringify({ email, password, action: 'signup' }),
        });

        // Save the token from the response
        if (data.token) {
            console.log('[API] Sign up successful, saving token');
            await this.saveToken(data.token);
            this.isAuthenticated = true;
        }

        return data;
    }

    async googleSignIn(token) {
        console.log('[API] Google sign in...');
        const data = await this.request('/api/auth/google', {
            method: 'POST',
            body: JSON.stringify({ token }),
        });

        // Save the token from the response
        if (data.token) {
            console.log('[API] Google sign in successful, saving token');
            await this.saveToken(data.token);
            this.isAuthenticated = true;
        }

        return data;
    }

    async createCheckoutSession() {
        return this.request('/api/stripe/checkout', {
            method: 'POST',
        });
    }

    async createPortalSession() {
        return this.request('/api/stripe/portal', {
            method: 'POST',
        });
    }

    async signOut() {
        try {
            console.log('[API] Signing out...');
            await this.removeToken();
            this.isAuthenticated = false;
            console.log('[API] Sign out successful');
        } catch (error) {
            console.error('[API] Sign out error:', error);
        }
    }
}

// Export singleton instance
window.clip2postAPI = new Clip2PostAPI();
