// Clip2Post Extension Popup Logic

// DOM Elements
let elements = {};
let state = {
    isAuthenticated: false,
    user: null,
    currentPlatform: null,
    lastGeneration: null,
};

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    initializeElements();
    setupEventListeners();
    checkAuth();
    loadHistory();

    // Auto-load last selected text from storage
    chrome.storage.local.get(['lastSelection'], (result) => {
        if (result.lastSelection && elements.inputText) {
            elements.inputText.value = result.lastSelection;
            // Clear the storage after loading
            chrome.storage.local.remove(['lastSelection']);
        }
    });
});

// Initialize DOM elements
function initializeElements() {
    elements = {
        // Views
        mainView: document.getElementById('mainView'),
        authView: document.getElementById('authView'),
        settingsView: document.getElementById('settingsView'),
        loadingState: document.getElementById('loadingState'),
        errorState: document.getElementById('errorState'),
        errorMessage: document.getElementById('errorMessage'),

        // Inputs
        inputText: document.getElementById('inputText'),
        platformBtns: document.querySelectorAll('.platform-btn'),

        // Output
        outputSection: document.getElementById('outputSection'),
        outputContent: document.getElementById('outputContent'),
        outputHashtags: document.getElementById('outputHashtags'),
        outputNotes: document.getElementById('outputNotes'),
        copyBtn: document.getElementById('copyBtn'),
        regenerateBtn: document.getElementById('regenerateBtn'),

        // History
        historyToggleBtn: document.getElementById('historyToggleBtn'),
        historySection: document.getElementById('historySection'),
        historyList: document.getElementById('historyList'),

        // Auth
        authEmail: document.getElementById('authEmail'),
        authPassword: document.getElementById('authPassword'),
        authSignInBtn: document.getElementById('authSignInBtn'),
        authSignUpBtn: document.getElementById('authSignUpBtn'),
        authError: document.getElementById('authError'),

        // Settings
        signedInState: document.getElementById('signedInState'),
        signedOutState: document.getElementById('signedOutState'),
        signInLink: document.getElementById('signInLink'),
        userEmail: document.getElementById('userEmail'),
    };
}

// Check authentication status
async function checkAuth() {
    try {
        const data = await window.clip2postAPI.checkAuth();

        if (data) {
            state.isAuthenticated = true;
            state.user = {
                email: data.email,
                plan: data.plan,
                quotaUsed: data.quotaUsed,
                quotaLimit: data.quotaLimit,
                subscriptionStatus: data.subscriptionStatus,
            };
        } else {
            state.isAuthenticated = false;
            state.user = null;
        }

        updateAuthUI();
    } catch (error) {
        console.error('Auth check error:', error);
        state.isAuthenticated = false;
        updateAuthUI();
    }
}

// Update UI based on auth state
function updateAuthUI() {
    if (state.isAuthenticated) {
        // Show main view
        if (elements.mainView) elements.mainView.classList.remove('hidden');
        if (elements.authView) elements.authView.classList.add('hidden');

        // Update account info in settings if visible
        if (state.user) {
            const usageText = document.getElementById('usageText');
            const usageBar = document.getElementById('usageBar');
            const planBadge = document.getElementById('planBadge');
            const accountEmail = document.getElementById('accountEmail');

            if (usageText) usageText.textContent = `${state.user.quotaUsed} / ${state.user.quotaLimit}`;
            if (usageBar) {
                const percentage = (state.user.quotaUsed / state.user.quotaLimit) * 100;
                usageBar.style.width = `${Math.min(percentage, 100)}%`;
            }
            if (planBadge) planBadge.textContent = state.user.plan === 'pro' ? 'PRO' : 'FREE';
            if (accountEmail) accountEmail.textContent = state.user.email;
        }

        // Settings view: show signed in state, hide signed out state
        if (elements.signedInState) elements.signedInState.classList.remove('hidden');
        if (elements.signedOutState) elements.signedOutState.classList.add('hidden');
        if (elements.userEmail) elements.userEmail.textContent = state.user?.email || '';
    } else {
        // Show auth view
        if (elements.mainView) elements.mainView.classList.add('hidden');
        if (elements.authView) elements.authView.classList.remove('hidden');

        // Settings view: hide signed in state, show signed out state
        if (elements.signedInState) elements.signedInState.classList.add('hidden');
        if (elements.signedOutState) elements.signedOutState.classList.remove('hidden');
    }
}

// Setup event listeners
function setupEventListeners() {
    console.log('Clip2Post: Setting up event listeners');

    // Settings button
    const settingsBtn = document.getElementById('settingsBtn');
    if (settingsBtn) {
        settingsBtn.addEventListener('click', () => {
            console.log('Settings button clicked');
            showSettings();
        });
    }

    // Back button
    const backBtn = document.getElementById('backBtn');
    if (backBtn) {
        backBtn.addEventListener('click', () => {
            console.log('Back button clicked');
            hideSettings();
        });
    }

    // Platform buttons
    if (elements.platformBtns) {
        elements.platformBtns.forEach((btn) => {
            btn.addEventListener('click', () => handlePlatformClick(btn));
        });
    }

    // Copy button
    if (elements.copyBtn) elements.copyBtn.addEventListener('click', handleCopy);

    // Regenerate button
    if (elements.regenerateBtn) {
        elements.regenerateBtn.addEventListener('click', () => {
            if (state.lastGeneration) {
                generateContent(state.lastGeneration.platform, state.lastGeneration.inputText);
            }
        });
    }

    // History toggle
    if (elements.historyToggleBtn) {
        elements.historyToggleBtn.addEventListener('click', toggleHistory);
    }

    // Auth buttons
    if (elements.authSignInBtn) {
        elements.authSignInBtn.addEventListener('click', () => handleInternalAuth('signin'));
    }
    if (elements.authSignUpBtn) {
        elements.authSignUpBtn.addEventListener('click', () => handleInternalAuth('signup'));
    }

    // Google Sign In
    const googleBtn = document.getElementById('googleSignInBtn');
    if (googleBtn) {
        googleBtn.addEventListener('click', handleGoogleSignIn);
    }

    // Sign Out button
    const signOutBtn = document.getElementById('signOutBtn');
    if (signOutBtn) {
        signOutBtn.addEventListener('click', async () => {
            await window.clip2postAPI.signOut();
            state.isAuthenticated = false;
            state.user = null;
            updateAuthUI();
            hideSettings();
        });
    }

    // Upgrade button
    const upgradeBtn = document.getElementById('upgradeBtn');
    if (upgradeBtn) {
        upgradeBtn.addEventListener('click', async () => {
            try {
                const { url } = await window.clip2postAPI.createCheckoutSession();
                if (url) chrome.tabs.create({ url });
            } catch (error) {
                console.error('Checkout error:', error);
                showError('Failed to create checkout session. Please try again.');
            }
        });
    }

    // Manage subscription button
    const manageSubBtn = document.getElementById('manageSubBtn');
    if (manageSubBtn) {
        manageSubBtn.addEventListener('click', async () => {
            try {
                const { url } = await window.clip2postAPI.createPortalSession();
                if (url) chrome.tabs.create({ url });
            } catch (error) {
                console.error('Portal error:', error);
                showError('Failed to open billing portal. Please try again.');
            }
        });
    }

    // Sign In link (in settings when logged out)
    if (elements.signInLink) {
        elements.signInLink.addEventListener('click', (e) => {
            e.preventDefault();
            hideSettings();
            showAuthView();
        });
    }
}

// Handle Google Sign In
async function handleGoogleSignIn() {
    setAuthLoading(true);
    hideAuthError();

    try {
        chrome.identity.getAuthToken({ interactive: true }, async (token) => {
            if (chrome.runtime.lastError || !token) {
                console.error('Google Auth Error:', chrome.runtime.lastError);
                showAuthError('Google sign-in cancelled or failed.');
                setAuthLoading(false);
                return;
            }

            try {
                // Exchange token for session
                await window.clip2postAPI.googleSignIn(token);
                await checkAuth(); // Refresh state

            } catch (err) {
                console.error('Backend Google Auth Error:', err);
                showAuthError('Google authentication failed.');
                // If invalid token, remove it so user can try again
                chrome.identity.removeCachedAuthToken({ token });
            } finally {
                setAuthLoading(false);
            }
        });
    } catch (e) {
        console.error('Identity API Error:', e);
        showAuthError('Chrome Identity API not available.');
        setAuthLoading(false);
    }
}

// Handle internal authentication (Sign In / Sign Up)
async function handleInternalAuth(mode) {
    const email = elements.authEmail.value.trim();
    const password = elements.authPassword.value;

    if (!email || !password) {
        showAuthError('Please enter both email and password.');
        return;
    }

    setAuthLoading(true);
    hideAuthError();

    try {
        let response;
        if (mode === 'signup') {
            response = await window.clip2postAPI.signUp(email, password);
        } else {
            response = await window.clip2postAPI.signIn(email, password);
        }

        // Success! The cookie should be set by the server
        await checkAuth(); // Refresh state

    } catch (error) {
        console.error('Auth error:', error);
        showAuthError(error.message || 'Authentication failed.');
    } finally {
        setAuthLoading(false);
    }
}

// Handle platform button click
async function handlePlatformClick(btn) {
    const platform = btn.dataset.platform;
    const inputText = elements.inputText.value.trim();

    if (!inputText) {
        showError('Please enter or select some text first.');
        return;
    }

    if (!state.isAuthenticated) {
        showError('Please sign in to generate content.');
        showAuthView();
        return;
    }

    // Update active state
    elements.platformBtns.forEach((b) => b.classList.remove('active'));
    btn.classList.add('active');
    state.currentPlatform = platform;

    await generateContent(platform, inputText);
}

// Generate content
async function generateContent(platform, inputText) {
    showLoading();
    hideError();

    try {
        const data = await window.clip2postAPI.generate(platform, inputText);

        state.lastGeneration = {
            platform,
            inputText,
            ...data,
            timestamp: Date.now(),
        };

        showOutput(data);
        saveToHistory(state.lastGeneration);

        // Update quota display
        if (state.user) {
            state.user.quotaUsed = state.user.quotaLimit - data.quotaRemaining;
            updateAuthUI();
        }

    } catch (error) {
        console.error('Generation detailed error:', {
            message: error.message,
            stack: error.stack,
            platform,
            isAuthenticated: state.isAuthenticated
        });
        showError(`Failed: ${error.message}`);

        // If auth error, show auth view
        if (error.message.includes('sign in') || error.message.includes('Session expired')) {
            state.isAuthenticated = false;
            updateAuthUI();
        }
    } finally {
        hideLoading();
    }
}

// Show output
function showOutput(data) {
    elements.outputSection.classList.remove('hidden');
    elements.outputContent.textContent = data.content;

    // Hashtags
    elements.outputHashtags.innerHTML = '';
    if (data.hashtags && data.hashtags.length > 0) {
        data.hashtags.forEach((tag) => {
            const span = document.createElement('span');
            span.className = 'hashtag';
            span.textContent = tag.startsWith('#') ? tag : `#${tag}`;
            elements.outputHashtags.appendChild(span);
        });
    }

    // Notes
    if (data.notes) {
        elements.outputNotes.textContent = data.notes;
        elements.outputNotes.classList.remove('hidden');
    } else {
        elements.outputNotes.classList.add('hidden');
    }
}

// Handle copy
async function handleCopy() {
    if (!state.lastGeneration) return;

    let textToCopy = state.lastGeneration.content;

    // Add hashtags for social platforms
    if (['x', 'linkedin', 'tiktok'].includes(state.lastGeneration.platform)) {
        if (state.lastGeneration.hashtags && state.lastGeneration.hashtags.length > 0) {
            const hashtagStr = state.lastGeneration.hashtags
                .map((t) => (t.startsWith('#') ? t : `#${t}`))
                .join(' ');
            textToCopy += '\n\n' + hashtagStr;
        }
    }

    try {
        await navigator.clipboard.writeText(textToCopy);
        const originalText = elements.copyBtn.textContent;
        elements.copyBtn.textContent = 'Copied!';
        setTimeout(() => {
            elements.copyBtn.textContent = originalText;
        }, 2000);
    } catch (err) {
        console.error('Failed to copy text: ', err);
    }
}

// History functions
function toggleHistory() {
    if (elements.historySection.classList.contains('hidden')) {
        elements.historySection.classList.remove('hidden');
        renderHistory();
    } else {
        elements.historySection.classList.add('hidden');
    }
}

function saveToHistory(item) {
    chrome.storage.local.get(['history'], (result) => {
        const history = result.history || [];
        history.unshift(item);
        // Keep last 10 items
        if (history.length > 10) history.pop();
        chrome.storage.local.set({ history });
        if (!elements.historySection.classList.contains('hidden')) {
            renderHistory();
        }
    });
}

function loadHistory() {
    chrome.storage.local.get(['history'], (result) => {
        if (result.history) {
            // Optional: Show indicator that history exists
        }
    });
}

function renderHistory() {
    chrome.storage.local.get(['history'], (result) => {
        const history = result.history || [];
        elements.historyList.innerHTML = '';

        if (history.length === 0) {
            elements.historyList.innerHTML = '<div class="history-empty">No history yet</div>';
            return;
        }

        history.forEach((item) => {
            const div = document.createElement('div');
            div.className = 'history-item';
            div.innerHTML = `
                <div class="history-meta">
                    <span class="history-platform">${item.platform}</span>
                    <span class="history-time">${formatTime(item.timestamp)}</span>
                </div>
                <div class="history-text">${item.content.substring(0, 60)}...</div>
            `;

            div.addEventListener('click', () => {
                state.lastGeneration = item;
                showOutput(item);

                // Update active state
                elements.platformBtns.forEach((b) => {
                    if (b.dataset.platform === item.platform) {
                        b.classList.add('active');
                    } else {
                        b.classList.remove('active');
                    }
                });
            });

            elements.historyList.appendChild(div);
        });
    });
}

function formatTime(isoString) {
    const date = new Date(isoString);
    const now = new Date();
    const diffMs = now - date;
    const diffMins = Math.floor(diffMs / 60000);
    const diffHours = Math.floor(diffMs / 3600000);

    if (diffMins < 1) return 'Just now';
    if (diffMins < 60) return `${diffMins}m ago`;
    if (diffHours < 24) return `${diffHours}h ago`;
    return date.toLocaleDateString();
}

// UI helpers
function showLoading() {
    if (elements.loadingState) elements.loadingState.classList.remove('hidden');
    if (elements.outputSection) elements.outputSection.classList.add('hidden');
}

function hideLoading() {
    if (elements.loadingState) elements.loadingState.classList.add('hidden');
}

function showError(message) {
    if (elements.errorMessage) elements.errorMessage.textContent = message;
    if (elements.errorState) elements.errorState.classList.remove('hidden');
}

function hideError() {
    if (elements.errorState) elements.errorState.classList.add('hidden');
}

function showSettings() {
    if (elements.settingsView) elements.settingsView.classList.remove('hidden');
    if (elements.mainView) elements.mainView.classList.add('hidden');
    if (elements.authView) elements.authView.classList.add('hidden');
    const settingsBtn = document.getElementById('settingsBtn');
    if (settingsBtn) settingsBtn.classList.add('active'); // Visual feedback
}

function hideSettings() {
    if (elements.settingsView) elements.settingsView.classList.add('hidden');
    const settingsBtn = document.getElementById('settingsBtn');
    if (settingsBtn) settingsBtn.classList.remove('active');

    if (state.isAuthenticated) {
        if (elements.mainView) elements.mainView.classList.remove('hidden');
    } else {
        if (elements.authView) elements.authView.classList.remove('hidden');
    }
}

function showAuthView() {
    if (elements.authView) elements.authView.classList.remove('hidden');
    if (elements.mainView) elements.mainView.classList.add('hidden');
    if (elements.settingsView) elements.settingsView.classList.add('hidden');
}

function showAuthError(message) {
    if (elements.authError) {
        elements.authError.textContent = message;
        elements.authError.classList.remove('hidden');
    }
}

function hideAuthError() {
    if (elements.authError) elements.authError.classList.add('hidden');
}

function setAuthLoading(loading) {
    if (elements.authSignInBtn) elements.authSignInBtn.disabled = loading;
    if (elements.authSignUpBtn) elements.authSignUpBtn.disabled = loading;
    if (elements.authSignInBtn) elements.authSignInBtn.textContent = loading ? '...' : 'Sign In';
}
